# Cargar las librerías necesarias
library(readxl)
library(dplyr)
library(caret)

df <- read_excel("atlas_mun.xlsx")

# Definir las columnas de identificación y las variables dependientes
id_columns <- c("ent", "mun", "sexo", "ponde_f")
dependent_vars <- c("pesotalla", "iemc")

# Convertir todas las columnas a formato numérico donde sea posible
df_numeric <- df %>%
  mutate(across(where(is.character), as.numeric)) %>%
  mutate(across(where(is.factor), as.numeric))

# Calcular la matriz de correlación
corr_matrix <- cor(df_numeric, use = "complete.obs")

# Identificar las variables perfectamente correlacionadas con las dependientes
perfect_corr <- function(var, threshold = 1.0) {
  corr <- abs(corr_matrix[, var])
  names(corr[corr == threshold])
}

perfect_corr_columns <- unique(c(
  perfect_corr("pesotalla"),
  perfect_corr("iemc")
))

# Eliminar variables perfectamente correlacionadas
df_cleaned <- df_numeric %>%
  select(-one_of(perfect_corr_columns))

# Filtrar las variables independientes, excluyendo las de identificación y dependientes
independent_vars <- setdiff(names(df_cleaned), c(id_columns, dependent_vars))

# Preparar los datos para el modelo
X <- df_cleaned %>%
  select(one_of(independent_vars))

y_pesotalla <- df$pesotalla
y_iemc <- df$iemc

# Crear y ajustar el modelo de regresión lineal múltiple para pesotalla
model_pesotalla <- lm(pesotalla ~ ., data = cbind(X, pesotalla = y_pesotalla))

# Crear y ajustar el modelo de regresión lineal múltiple para iemc
model_iemc <- lm(iemc ~ ., data = cbind(X, iemc = y_iemc))

# Interpretación de los coeficientes del modelo para pesotalla
summary(model_pesotalla)

# Interpretación de los coeficientes del modelo para iemc
summary(model_iemc)

# Validación cruzada para pesotalla
set.seed(123)
train_control <- trainControl(method = "cv", number = 10)
model_cv_pesotalla <- train(pesotalla ~ ., data = cbind(X, pesotalla = y_pesotalla), method = "lm", trControl = train_control)
model_cv_pesotalla$results

# Validación cruzada para iemc
model_cv_iemc <- train(iemc ~ ., data = cbind(X, iemc = y_iemc), method = "lm", trControl = train_control)
model_cv_iemc$results

### STEPFORWARD

# Cargar la librería olsrr
library(olsrr)

# Regresión stepwise forward para pesotalla
model_full_pesotalla <- lm(pesotalla ~ ., data = cbind(X, pesotalla = y_pesotalla))
stepwise_model_pesotalla <- ols_step_forward_p(model_full_pesotalla)
summary(stepwise_model_pesotalla$model)

# Evaluación del modelo stepwise para pesotalla
mse_stepwise_pesotalla <- mean(stepwise_model_pesotalla$model$residuals^2)
r2_stepwise_pesotalla <- summary(stepwise_model_pesotalla$model)$r.squared

# Regresión stepwise forward para iemc
model_full_iemc <- lm(iemc ~ ., data = cbind(X, iemc = y_iemc))
stepwise_model_iemc <- ols_step_forward_p(model_full_iemc)
summary(stepwise_model_iemc$model)

# Evaluación del modelo stepwise para iemc
mse_stepwise_iemc <- mean(stepwise_model_iemc$model$residuals^2)
r2_stepwise_iemc <- summary(stepwise_model_iemc$model)$r.squared

# Resultados del modelo stepwise
list(
  mse_stepwise_pesotalla = mse_stepwise_pesotalla,
  r2_stepwise_pesotalla = r2_stepwise_pesotalla,
  mse_stepwise_iemc = mse_stepwise_iemc,
  r2_stepwise_iemc = r2_stepwise_iemc
)
